import numpy as np
from matplotlib import pyplot as plt

######################################################################################################################################################
#   Code for producing Figure 5--Figure Supplement 1.
#   The following data is used in this figure:
#   -- theoryStepRatio_const.txt, backward-to-forward step ratio predicted by the free diffusion model (\mu_c = 0 k_B T) as a function of load force
#   -- kad_ratio.txt, backward-to-forward step ratio measured by Kad et al. (2008)
#
#   -- theoryRunLength_const.txt, mean run length predicted by the free diffusion model (\mu_c = 0 k_B T) as a function of load force
#   -- clemen_runlength.txt, mean run lengths measured by Clemen et al. (2005)
#   -- Zero force run lengths measured by Sakamoto et al. (2000), Baker et al. (2004), and Pierobon et al. (2009) are included in code below
#
#   -- theoryRunVelocity_const.txt, mean run velocity predicted by the free diffusion model (\mu_c = 0 k_B T) as a function of load force
#   -- theoryRunVelocityLowATP_const.txt, mean run velocity predicted by the free diffusion model (\mu_c = 0 k_B T) as a function of load force
#                                                   with low ATP adjustment to the trailing leg dissociation rate
#   -- clemen_velocity.txt, mean run velocities measured by Clemen et al. (2005)
#   -- gebhardt_velocity.txt, mean run velocities measured by Gebhardt et al. (2006)
#   -- kad_velocity.txt, mean run velocities measured by Kad et al. (2008)
#   -- mehta_velocity.txt, mean run velocities measured by Mehta et al. (1999)
#   -- uemura_velocity.txt, mean run velocities measured by Uemura et al. (2004)

######################################################################################################################################################


axisLabelSize = 18
lw = 3
msize = 9
legendTxt = 16
mkWidth = 2
handle = 0.75
tickLen=6
tickLabel=13

params = {'backend': 'ps'}

plt.rcParams.update(params)

F, stepRatioTheory = np.transpose(np.loadtxt('theory_force_response/theoryStepRatio.txt'))
FkadRatio, stepRatioKad = np.transpose(np.loadtxt('experimental_force_response/kad_ratio.txt', delimiter=','))

_, runLengthTheory = np.transpose(np.loadtxt('theory_force_response/theoryRunLength.txt'))
Fclem, runLengthClemen = np.transpose(np.loadtxt('experimental_force_response/clemen_runlength.txt', delimiter=','))

_, runVelocityTheory = np.transpose(np.loadtxt('theory_force_response/theoryRunVelocity.txt'))
_, runVelocityTheoryLowATP = np.transpose(np.loadtxt('theory_force_response/theoryRunVelocityLowATP.txt'))
FclemVel, runVelocityClemen = np.transpose(np.loadtxt('experimental_force_response/clemen_velocity.txt', delimiter=','))
FgebVel, runVelocitygebhardt = np.transpose(np.loadtxt('experimental_force_response/gebhardt_velocity.txt', delimiter=','))
FkadVel, runVelocitykad = np.transpose(np.loadtxt('experimental_force_response/kad_velocity.txt', delimiter=','))
FmehtaVel, runVelocitymehta = np.transpose(np.loadtxt('experimental_force_response/mehta_velocity.txt', delimiter=','))
FuemuraVel, runVelocityuemura = np.transpose(np.loadtxt('experimental_force_response/uemura_velocity.txt', delimiter=','))




fig, axs = plt.subplots(3, 1, figsize=(9, 17), sharex=True, squeeze=True)
plt.subplots_adjust(wspace=0, hspace=0)



test = axs[0].plot(FkadRatio, stepRatioKad, 'Dm', markerfacecolor='none', markersize=msize, markeredgewidth=mkWidth)
axs[0].plot(F,stepRatioTheory, 'grey', linewidth=lw)
axs[0].plot([1.89,1.89], [0, 1], 'r--', linewidth=lw-1)
axs[0].plot([0,4], [1, 1], 'r--', linewidth=lw-1)
axs[0].set_ylim([0,1.2])
axs[0].set_xlim([0,3.5])
axs[0].set_ylabel('Step Ratio $\mathcal{P}_b/\mathcal{P}_f$', fontsize=axisLabelSize, x=0)
axs[0].text(-0.07,1.,'A',transform=axs[0].transAxes, horizontalalignment='right', verticalalignment='center', size=axisLabelSize+4)
axs[0].yaxis.set_label_coords(-0.07,0.5)
axs[0].tick_params('both', top=True, right=True,direction='in',length=tickLen,labelsize=tickLabel, width=2)
# axs[0].set_title('Free Diffusion Theory ($\mu_c=0$)', size=axisLabelSize+4)
for axis in ['top','bottom','left','right']:
  axs[0].spines[axis].set_linewidth(2)
axs[0].text(1.95, 0.2, '$F_{\\regular{stall}}$', fontsize=axisLabelSize+2, color='red')
# Sakamoto:  2.39, 1.76, 1.17, 0.66
# Baker:  1.3, 0.8
# Pierobon in vitro:  1.3
# Pierobon in vivo:  2.2

delta = 0.0125
axs[1].plot(Fclem, runLengthClemen/1000, 'sc', markerfacecolor='none', markersize=msize, markeredgewidth=mkWidth)
axs[1].plot(np.zeros(4)+delta, [0.66, 1.17, 1.76, 2.39],'^b', [delta, delta], [0.8, 1.3], 'pr',
            [delta], [1.3], '+g', [delta], [2.2], '+m', markerfacecolor='none', markersize=msize+0.5,
            markeredgewidth=mkWidth)
axs[1].plot(F,runLengthTheory, 'grey', linewidth=lw)
axs[1].plot([1.89,1.89], [0, 1.47], 'r--', [1.89,1.89],[2.38,3], 'r--', linewidth=lw-1)
axs[1].set_ylim([0,2.48])
axs[1].set_ylabel('Run Length $\\regular{z_{run} \, \, [\mu m]}$', fontsize=axisLabelSize)
axs[1].text(-0.07,1.,'B',transform=axs[1].transAxes, horizontalalignment='right', verticalalignment='center', size=axisLabelSize+4)
axs[1].yaxis.set_label_coords(-0.07,0.5)
axs[1].tick_params('both', top=True, right=True,direction='in',length=tickLen,labelsize=tickLabel, width=2)
for axis in ['top','bottom','left','right']:
  axs[1].spines[axis].set_linewidth(2)

axs[2].plot(FclemVel, runVelocityClemen, 'sc', FmehtaVel, runVelocitymehta, 'ob',
            FuemuraVel, runVelocityuemura, '^r',FkadVel, runVelocitykad, 'Dm', FgebVel, runVelocitygebhardt, 'xg',
            markerfacecolor='none', markersize=msize, markeredgewidth=mkWidth)
axs[2].plot(F,runVelocityTheory, 'grey', linewidth=lw)
axs[2].plot(F, runVelocityTheoryLowATP, 'grey',linewidth=lw+1, dashes=(1, 1))
axs[2].set_ylim([-100,480])
axs[2].set_ylabel('Run Velocity $\\regular{v_{run} \, \, [nm/s]}$', fontsize=axisLabelSize)
axs[2].set_xlabel('F [pN]', fontsize=axisLabelSize)
axs[2].text(-0.07,1.,'C',transform=axs[2].transAxes, horizontalalignment='right', verticalalignment='center', size=axisLabelSize+4)
axs[2].yaxis.set_label_coords(-0.07,0.5)
axs[2].tick_params('both', top=True, right=True,direction='in',length=tickLen,labelsize=tickLabel, width=2)
axs[2].tick_params('x', pad=8)
for axis in ['top','bottom','left','right']:
  axs[2].spines[axis].set_linewidth(2)

axs[0].legend(['Kad $et \, \, al.$ (100 $\mu$M ATP)', 'Theory'], loc='center left', frameon=False, fontsize=legendTxt, handlelength=handle)
axs[1].legend(['Clemen $et \, \, al.$ (2 mM ATP, 25 mM KCl)', 'Sakamoto $et \, \, al.$ (2 mM ATP, 150-400 mM KCl)',
               'Baker $et \, \, al.$ (1 mM ATP, 25-100 mM KCl)', 'Pierbon $et \, \, al.$ (2 mM ATP, 25 mM KCl)',
               'Pierbon $et \, \, al.$ ($in \, \, vivo$)', 'Theory'],loc='upper right', frameon=False, fontsize=legendTxt, handlelength=handle)
axs[2].legend(['Clemen $et \, \, al.$ (2 mM)', 'Mehta $et \, \, al.$ (1 mM)', 'Uemura $et \, \, al.$ (1 mM)',
               'Kad $et \, \, al.$ (100 $\mu$M)', 'Gebhardt $et \, \, al.$ (1 $\mu$M)', 'Theory',
               'Theory (low ATP)'],loc='upper right', frameon=False, fontsize=legendTxt, handlelength=handle)



plt.savefig("force_dependence_free_diffusion.pdf",bbox_inches='tight')

plt.show()